<?php
require_once 'config/database.php';
require_once 'config/auth.php';
require_once 'config/file_upload.php';

// Verificar autenticación
requerirAutenticacion();

// Procesar acciones CRUD
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $pdo = conectarDB();
    
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'create':
                // Procesar archivo PDF
                $archivoSubido = null;
                $thumbnailGenerado = null;
                
                if (isset($_FILES['archivo_pdf']) && $_FILES['archivo_pdf']['error'] === UPLOAD_ERR_OK) {
                    $resultadoSubida = subirArchivoPDF($_FILES['archivo_pdf']);
                    if ($resultadoSubida['success']) {
                        $archivoSubido = $resultadoSubida['archivo'];
                        $thumbnailGenerado = generarThumbnailPDF($resultadoSubida['ruta'], $archivoSubido);
                    } else {
                        $error = implode(', ', $resultadoSubida['errores']);
                        break;
                    }
                } else {
                    $error = "Debe seleccionar un archivo PDF";
                    break;
                }
                
                $stmt = $pdo->prepare("INSERT INTO repositorio (categoria, autor, titulo, descripcion, fecha_publicado, url_documento, thumbnails) VALUES (?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([
                    $_POST['categoria'],
                    $_POST['autor'],
                    $_POST['titulo'],
                    $_POST['descripcion'],
                    $_POST['fecha_publicado'],
                    $archivoSubido,
                    $thumbnailGenerado
                ]);
                $mensaje = "Repositorio creado exitosamente";
                break;
                
            case 'update':
                // Obtener datos actuales
                $stmt = $pdo->prepare("SELECT url_documento, thumbnails FROM repositorio WHERE id = ?");
                $stmt->execute([$_POST['id']]);
                $datosActuales = $stmt->fetch(PDO::FETCH_ASSOC);
                
                $archivoSubido = $datosActuales['url_documento'];
                $thumbnailGenerado = $datosActuales['thumbnails'];
                
                // Procesar nuevo archivo si se subió uno
                if (isset($_FILES['archivo_pdf']) && $_FILES['archivo_pdf']['error'] === UPLOAD_ERR_OK) {
                    $resultadoSubida = subirArchivoPDF($_FILES['archivo_pdf']);
                    if ($resultadoSubida['success']) {
                        // Eliminar archivo anterior
                        eliminarArchivo($datosActuales['url_documento'], $datosActuales['thumbnails']);
                        
                        $archivoSubido = $resultadoSubida['archivo'];
                        $thumbnailGenerado = generarThumbnailPDF($resultadoSubida['ruta'], $archivoSubido);
                    } else {
                        $error = implode(', ', $resultadoSubida['errores']);
                        break;
                    }
                }
                
                $stmt = $pdo->prepare("UPDATE repositorio SET categoria=?, autor=?, titulo=?, descripcion=?, fecha_publicado=?, url_documento=?, thumbnails=? WHERE id=?");
                $stmt->execute([
                    $_POST['categoria'],
                    $_POST['autor'],
                    $_POST['titulo'],
                    $_POST['descripcion'],
                    $_POST['fecha_publicado'],
                    $archivoSubido,
                    $thumbnailGenerado,
                    $_POST['id']
                ]);
                $mensaje = "Repositorio actualizado exitosamente";
                break;
                
            case 'delete':
                // Obtener datos del archivo antes de eliminar
                $stmt = $pdo->prepare("SELECT url_documento, thumbnails FROM repositorio WHERE id = ?");
                $stmt->execute([$_POST['id']]);
                $datosArchivo = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Eliminar archivo físico
                if ($datosArchivo) {
                    eliminarArchivo($datosArchivo['url_documento'], $datosArchivo['thumbnails']);
                }
                
                // Eliminar registro de la base de datos
                $stmt = $pdo->prepare("DELETE FROM repositorio WHERE id=?");
                $stmt->execute([$_POST['id']]);
                $mensaje = "Repositorio eliminado exitosamente";
                break;
        }
    }
}

// Obtener repositorio para editar
$repositorio_editar = null;
if (isset($_GET['editar']) && is_numeric($_GET['editar'])) {
    $pdo = conectarDB();
    $stmt = $pdo->prepare("SELECT * FROM repositorio WHERE id = ?");
    $stmt->execute([$_GET['editar']]);
    $repositorio_editar = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Obtener todos los repositorios
$pdo = conectarDB();
$stmt = $pdo->query("SELECT * FROM repositorio ORDER BY fecha_publicado DESC");
$repositorios = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestión de Repositorios</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100 min-h-screen">
    <!-- Header con información del usuario -->
    <div class="bg-white shadow-sm border-b">
        <div class="container mx-auto px-4 py-4">
            <div class="flex justify-between items-center">
                <div>
                    <h1 class="text-2xl font-bold text-gray-800">Gestión de Repositorios</h1>
                    <p class="text-sm text-gray-600">
                        Usuario: <?php echo htmlspecialchars(obtenerUsuarioActual()['email']); ?>
                    </p>
                </div>
                <div class="flex items-center space-x-4">
                    <span class="text-sm text-gray-500" id="tiempo-sesion">
                        Sesión válida por: <?php 
                        $tiempo_restante = 7200 - (time() - $_SESSION['login_time']);
                        $horas = floor($tiempo_restante / 3600);
                        $minutos = floor(($tiempo_restante % 3600) / 60);
                        echo $horas . 'h ' . $minutos . 'm';
                        ?>
                    </span>
                    <a href="logout.php" 
                       class="bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded-md text-sm transition duration-200">
                        Cerrar Sesión
                    </a>
                </div>
            </div>
        </div>
    </div>
    
    <div class="container mx-auto px-4 py-8">
        
        <!-- Mensaje de confirmación -->
        <?php if (isset($mensaje)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo htmlspecialchars($mensaje); ?>
            </div>
        <?php endif; ?>
        
        <!-- Mensaje de error -->
        <?php if (isset($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>
        
        <!-- Formulario de creación/edición -->
        <div class="bg-white rounded-lg shadow-md p-6 mb-8">
            <h2 class="text-xl font-semibold mb-4">
                <?php echo $repositorio_editar ? 'Editar Repositorio' : 'Nuevo Repositorio'; ?>
            </h2>
            
            <form method="POST" enctype="multipart/form-data" class="space-y-4">
                <input type="hidden" name="action" value="<?php echo $repositorio_editar ? 'update' : 'create'; ?>">
                <?php if ($repositorio_editar): ?>
                    <input type="hidden" name="id" value="<?php echo htmlspecialchars($repositorio_editar['id']); ?>">
                <?php endif; ?>
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Categoría</label>
                        <input type="text" name="categoria" required 
                               value="<?php echo htmlspecialchars($repositorio_editar['categoria'] ?? ''); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Autor</label>
                        <input type="text" name="autor" required 
                               value="<?php echo htmlspecialchars($repositorio_editar['autor'] ?? ''); ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Título</label>
                    <input type="text" name="titulo" required 
                           value="<?php echo htmlspecialchars($repositorio_editar['titulo'] ?? ''); ?>"
                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                </div>
                
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Descripción</label>
                    <textarea name="descripcion" rows="3" 
                              class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"><?php echo htmlspecialchars($repositorio_editar['descripcion'] ?? ''); ?></textarea>
                </div>
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Fecha de Publicación</label>
                        <input type="date" name="fecha_publicado" required 
                               value="<?php echo $repositorio_editar['fecha_publicado'] ?? ''; ?>"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">
                            Archivo PDF 
                            <span class="text-red-500">*</span>
                            <span class="text-sm text-gray-500">(Máximo 50MB)</span>
                        </label>
                        <input type="file" name="archivo_pdf" accept=".pdf" 
                               <?php echo !$repositorio_editar ? 'required' : ''; ?>
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                        
                        <?php if ($repositorio_editar && $repositorio_editar['url_documento']): ?>
                            <div class="mt-2 p-2 bg-gray-100 rounded text-sm">
                                <p class="text-gray-600">Archivo actual: <strong><?php echo htmlspecialchars($repositorio_editar['url_documento']); ?></strong></p>
                                <p class="text-gray-500">Seleccione un nuevo archivo solo si desea reemplazarlo</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="flex gap-2">
                    <button type="submit" 
                            class="bg-blue-500 hover:bg-blue-600 text-white px-6 py-2 rounded-md transition duration-200">
                        <?php echo $repositorio_editar ? 'Actualizar' : 'Crear'; ?>
                    </button>
                    
                    <?php if ($repositorio_editar): ?>
                        <a href="gestion_repositorios.php" 
                           class="bg-gray-500 hover:bg-gray-600 text-white px-6 py-2 rounded-md transition duration-200">
                            Cancelar
                        </a>
                    <?php endif; ?>
                </div>
            </form>
        </div>
        
        <!-- Tabla de repositorios -->
        <div class="bg-white rounded-lg shadow-md overflow-hidden">
            <h2 class="text-xl font-semibold p-6 border-b">Lista de Repositorios</h2>
            
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ID</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Thumbnail</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Categoría</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Autor</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Título</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Fecha</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Acciones</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        <?php foreach ($repositorios as $repo): ?>
                            <tr>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($repo['id']); ?></td>
                                <td class="px-6 py-4 whitespace-nowrap">
                                    <?php if ($repo['thumbnails'] && file_exists(THUMBNAIL_DIR . $repo['thumbnails'])): ?>
                                        <img src="<?php echo THUMBNAIL_DIR . htmlspecialchars($repo['thumbnails']); ?>" 
                                             alt="Thumbnail" class="w-16 h-16 object-cover rounded border">
                                    <?php else: ?>
                                        <div class="w-16 h-16 bg-gray-200 rounded border flex items-center justify-center">
                                            <span class="text-gray-500 text-xs">PDF</span>
                                        </div>
                                    <?php endif; ?>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($repo['categoria']); ?></td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($repo['autor']); ?></td>
                                <td class="px-6 py-4 text-sm text-gray-900"><?php echo htmlspecialchars($repo['titulo']); ?></td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo date('d/m/Y', strtotime($repo['fecha_publicado'])); ?></td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                    <div class="flex flex-col space-y-1">
                                        <a href="detalle_repositorio.php?id=<?php echo $repo['id']; ?>" 
                                           class="text-blue-600 hover:text-blue-900">Ver</a>
                                        <a href="gestion_repositorios.php?editar=<?php echo $repo['id']; ?>" 
                                           class="text-green-600 hover:text-green-900">Editar</a>
                                        <?php if ($repo['url_documento'] && file_exists(UPLOAD_DIR . $repo['url_documento'])): ?>
                                            <a href="<?php echo UPLOAD_DIR . htmlspecialchars($repo['url_documento']); ?>" 
                                               target="_blank" class="text-purple-600 hover:text-purple-900">Descargar</a>
                                        <?php endif; ?>
                                        <form method="POST" style="display: inline;" onsubmit="return confirm('¿Estás seguro de eliminar este repositorio?')">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="id" value="<?php echo $repo['id']; ?>">
                                            <button type="submit" class="text-red-600 hover:text-red-900">Eliminar</button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <!-- Enlaces a otras páginas -->
        <div class="mt-8 flex gap-4">
            <a href="buscar_repositorios.php" 
               class="bg-green-500 hover:bg-green-600 text-white px-6 py-2 rounded-md transition duration-200">
                Buscar Repositorios
            </a>
            <a href="regenerar_thumbnails.php" 
               class="bg-purple-500 hover:bg-purple-600 text-white px-6 py-2 rounded-md transition duration-200">
                Regenerar Thumbnails
            </a>
        </div>
    </div>
    
    <script>
        // Actualizar tiempo de sesión cada minuto
        function actualizarTiempoSesion() {
            const elemento = document.getElementById('tiempo-sesion');
            if (!elemento) return;
            
            const tiempoTexto = elemento.textContent;
            const match = tiempoTexto.match(/(\d+)h (\d+)m/);
            
            if (match) {
                let horas = parseInt(match[1]);
                let minutos = parseInt(match[2]);
                
                minutos--;
                if (minutos < 0) {
                    minutos = 59;
                    horas--;
                }
                
                if (horas < 0) {
                    // Sesión expirada, redirigir a login
                    window.location.href = 'login.php';
                    return;
                }
                
                elemento.textContent = `Sesión válida por: ${horas}h ${minutos}m`;
            }
        }
        
        // Actualizar cada minuto
        setInterval(actualizarTiempoSesion, 60000);
    </script>
</body>
</html>
